<?php
// core/AccountController.php

require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/Logger.php';
require_once __DIR__ . '/NotificationController.php';

class AccountController {

    /**
     * إنشاء أو تحديث حساب طالب
     */
    public static function createOrUpdateAccount($student_id, $total_fee, $paid = 0, $due_date = null, $admin_id = null, $installment_count = 1) {
        global $conn;

        $acc = self::getAccount($student_id); // Auto-create يعمل هنا

        $remaining = $total_fee - $paid;

        // توزيع الدفعات
        $installments = self::generateInstallments($total_fee, $installment_count);

        $stmt2 = $conn->prepare("
            UPDATE accounts 
            SET total_fee=?, paid=?, remaining=?, due_date=?,
                installment1=?, installment2=?, installment3=?,
                installment4=?, installment5=?, installment6=?,
                last_update=NOW()
            WHERE student_id=?");

        $ok = $stmt2->execute([
            $total_fee, $paid, $remaining, $due_date,
            $installments[0], $installments[1], $installments[2],
            $installments[3], $installments[4], $installments[5],
            $student_id
        ]);

        if ($ok) {
            Logger::log($admin_id, $student_id, "UPDATE_ACCOUNT", "تحديث/إنشاء حساب مالي");
        }

        return $ok;
    }

    /**
     * توزيع الدفعات (1–6 دفعات)
     */
    private static function generateInstallments($total, $count) {
        $installments = [0,0,0,0,0,0];

        if ($count <= 1) {
            $installments[0] = $total;
            return $installments;
        }

        $per = intval($total / $count);

        for ($i = 0; $i < $count; $i++) {
            $installments[$i] = $per;
        }

        // معالجة المتبقي من القسمة
        $remain = $total - ($per * $count);
        if ($remain > 0) {
            $installments[0] += $remain;
        }

        return $installments;
    }

    /**
     * جلب حساب طالب (Auto Create)
     */
    public static function getAccount($student_id) {
        global $conn;

        // جلب الحساب المالي
        $stmt = $conn->prepare("SELECT * FROM accounts WHERE student_id=? LIMIT 1");
        $stmt->execute([$student_id]);
        $acc = $stmt->fetch(PDO::FETCH_ASSOC);

        // إذا لا يوجد حساب → إنشاؤه تلقائياً
        if (!$acc) {

            $conn->prepare("
                INSERT INTO accounts 
                (student_id, total_fee, paid, remaining, level, payment_plan,
                 installment1, installment2, installment3, installment4, installment5, installment6)
                VALUES (?, 0, 0, 0, 'المرحلة الأولى', 'دفعة واحدة', 0,0,0,0,0,0)
            ")->execute([$student_id]);

            // إعادة جلب الحساب الجديد
            $stmt = $conn->prepare("SELECT * FROM accounts WHERE student_id=? LIMIT 1");
            $stmt->execute([$student_id]);
            $acc = $stmt->fetch(PDO::FETCH_ASSOC);
        }

        return $acc;
    }

    /**
     * إضافة دفعة مالية
     */
    public static function addPayment($student_id, $amount, $receipt_number, $receipt_image = null, $notes = "", $admin_id = null) {
        global $conn;

        // حفظ الدفعة
        $stmt = $conn->prepare("
            INSERT INTO payments (student_id, amount, receipt_number, receipt_image, notes)
            VALUES (?,?,?,?,?)
        ");
        $ok = $stmt->execute([$student_id, $amount, $receipt_number, $receipt_image, $notes]);

        if (!$ok) return false;

        // تحديث الحساب المالي
        $acc = self::getAccount($student_id); // Auto-create يعمل هنا أيضاً

        $paid = $acc['paid'] + $amount;
        $total_fee = $acc['total_fee'];
        $remaining = $total_fee - $paid;

        $stmt2 = $conn->prepare("UPDATE accounts SET paid=?, remaining=?, last_update=NOW() WHERE student_id=?");
        $stmt2->execute([$paid, $remaining, $student_id]);

        // سجل العملية
        Logger::log($admin_id, $student_id, "ADD_PAYMENT", "إضافة دفعة بقيمة $amount");

        // ===== إرسال إشعار واتساب =====
        $student = self::getStudentData($student_id);

        if ($student) {
            $template = self::buildPaymentMessage($student, $amount, $paid, $remaining);
            NotificationController::sendToStudent($student, $template);
        }

        return true;
    }

    /**
     * رسالة الدفع
     */
    private static function buildPaymentMessage($student, $amount, $paid, $remaining) {

        return "
📢 *إشعار دفع جديد – جامعة النبراس*

👤 الطالب: *{$student['full_name']}*
🏛 القسم: *{$student['college_name']}*

💵 الدفعة المستلمة: *" . number_format($amount) . " د.ع*
💰 مجموع المدفوع: *" . number_format($paid) . " د.ع*
💸 المتبقي: *" . number_format($remaining) . " د.ع*

📅 التاريخ: " . date("Y-m-d H:i") ."
";
    }

    /**
     * جلب بيانات طالب كاملة (استخدام للواتساب)
     */
    public static function getStudentData($student_id) {
        global $conn;
        $stmt = $conn->prepare("
            SELECT s.*, a.total_fee, a.paid, a.remaining, a.due_date 
            FROM students s
            LEFT JOIN accounts a ON a.student_id = s.id
            WHERE s.id = ?
        ");
        $stmt->execute([$student_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * جلب جميع الدفعات
     */
    public static function getPayments($student_id) {
        global $conn;
        $stmt = $conn->prepare("
            SELECT * FROM payments WHERE student_id=?
            ORDER BY paid_at DESC
        ");
        $stmt->execute([$student_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * تحديث تاريخ الاستحقاق
     */
    public static function updateDueDate($student_id, $new_due_date, $reason, $admin_id = null) {
        global $conn;

        $stmt = $conn->prepare("
            UPDATE accounts SET due_date=?, last_update=NOW()
            WHERE student_id=?
        ");
        $ok = $stmt->execute([$new_due_date, $student_id]);

        if ($ok) {
            Logger::log($admin_id, $student_id, "UPDATE_DUE_DATE", "تم تمديد الاستحقاق: $reason");
        }

        return $ok;
    }
}