<?php
// core/NotificationController.php

require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../config/whatsapp.php';
require_once __DIR__ . '/Logger.php';

class NotificationController {

    /**
     * إرسال رسالة واتساب لطالب
     */
    public static function sendToStudent($student, $message_template) {
        global $conn;

        // التحقق من وجود ID حقيقي للطالب
        $student_id = $student['id'] ?? $student['student_id'] ?? null;
        if (!$student_id) return false;

        // جلب رقم الهاتف
        $phone = self::cleanPhone($student['phone'] ?? null);
        if (!$phone) {
            self::saveLog($student_id, $message_template, 'invalid_phone');
            return false;
        }

        // عملية الإرسال
        $result = false;
        try {
            $result = sendWhatsapp($phone, $message_template);
        } catch (Exception $e) {
            $result = false;
        }

        // حفظ سجل الإشعار
        $status = $result ? 'sent' : 'failed';

        $stmt = $conn->prepare("
            INSERT INTO notifications (student_id, phone, message, status, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([$student_id, $phone, $message_template, $status]);

        return $result;
    }


    /**
     * تنظيف رقم الهاتف ليكون صالح للإرسال
     */
    private static function cleanPhone($phone) {

        if (!$phone) return null;

        // إزالة المسافات والرموز
        $clean = preg_replace('/[^0-9]/', '', $phone);

        // دعم +964
        if (strpos($phone, "+964") !== false) {
            $clean = "964" . substr($clean, -10);
        }

        // دعم 0770… إلخ
        if (strlen($clean) == 11 && substr($clean, 0, 1) == "0") {
            $clean = "964" . substr($clean, 1);
        }

        // دعم 7xxx
        if (strlen($clean) == 10) {
            $clean = "964" . $clean;
        }

        // الرقم النهائي يجب أن يكون 13 خانة: 9647xxxxxxxx
        if (strlen($clean) != 13) return null;

        return $clean;
    }


    /**
     * استبدال المتغيرات داخل الرسالة
     */
    public static function buildMessage($student, $template) {

        $vars = [
            '{name}'       => $student['full_name'] ?? '',
            '{phone}'      => $student['phone'] ?? '',
            '{college}'    => $student['college_name'] ?? '',
            '{university}' => $student['university_name'] ?? '',
            '{gender}'     => $student['gender'] ?? '',
            '{level}'      => $student['level'] ?? '',
            '{total_fee}'  => number_format($student['total_fee'] ?? 0),
            '{paid}'       => number_format($student['paid'] ?? 0),
            '{remaining}'  => number_format($student['remaining'] ?? 0),
            '{due_date}'   => $student['due_date'] ?? '—',
            '{installment1}' => $student['installment1'] ?? 0,
            '{installment2}' => $student['installment2'] ?? 0,
            '{installment3}' => $student['installment3'] ?? 0,
            '{installment4}' => $student['installment4'] ?? 0,
            '{installment5}' => $student['installment5'] ?? 0,
            '{installment6}' => $student['installment6'] ?? 0,
        ];

        return str_replace(array_keys($vars), array_values($vars), $template);
    }


    /**
     * إرسال رسالة باستخدام Template من قاعدة البيانات
     */
    public static function sendTemplate($student, $template_key) {
        global $conn;

        $stmt = $conn->prepare("SELECT template_text FROM whatsapp_templates WHERE template_key=? LIMIT 1");
        $stmt->execute([$template_key]);
        $tpl = $stmt->fetchColumn();

        if (!$tpl) return false;

        $message = self::buildMessage($student, $tpl);
        return self::sendToStudent($student, $message);
    }


    /**
     * تسجيل فشل الإرسال إذا رقم الطالب خطأ
     */
    private static function saveLog($student_id, $msg, $status) {
        global $conn;

        $stmt = $conn->prepare("
            INSERT INTO notifications (student_id, message, status, created_at)
            VALUES (?, ?, ?, NOW())
        ");
        $stmt->execute([$student_id, $msg, $status]);
    }

}