<?php
// core/ReportController.php – B2Pro Ultra Edition

require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/AccountController.php';

class ReportController
{

    /**
     * جلب جميع الدفعات + بيانات الطالب + الحساب
     */
    public static function allPayments()
    {
        global $conn;

        $sql = "
            SELECT 
                p.*,
                s.full_name, s.gender, s.college_name, s.university_name, s.phone,
                a.total_fee, a.paid, a.remaining, a.level
            FROM payments p
            INNER JOIN students s ON s.id = p.student_id
            LEFT JOIN accounts a ON a.student_id = s.id
            ORDER BY p.paid_at DESC
        ";

        return $conn->query($sql)->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * جلب الطلاب المتأخرين (remaining > 0)
     */
    public static function lateStudents()
    {
        global $conn;

        $sql = "
            SELECT 
                s.*, 
                a.total_fee, a.paid, a.remaining, a.level, a.due_date
            FROM accounts a
            INNER JOIN students s ON s.id = a.student_id
            WHERE a.remaining > 0
            ORDER BY a.remaining DESC
        ";

        return $conn->query($sql)->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * جلب حسب المرحلة
     */
    public static function byLevel($level)
    {
        global $conn;

        $stmt = $conn->prepare("
            SELECT s.*, a.total_fee, a.paid, a.remaining, a.level
            FROM accounts a
            INNER JOIN students s ON s.id = a.student_id
            WHERE a.level = ?
        ");
        $stmt->execute([$level]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * جلب حسب الجنس
     */
    public static function byGender($gender)
    {
        global $conn;

        $stmt = $conn->prepare("
            SELECT s.*, a.total_fee, a.paid, a.remaining, a.level
            FROM accounts a
            INNER JOIN students s ON s.id = a.student_id
            WHERE s.gender = ?
        ");
        $stmt->execute([$gender]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * جلب حسب الكلية
     */
    public static function byCollege($college)
    {
        global $conn;

        $stmt = $conn->prepare("
            SELECT s.*, a.total_fee, a.paid, a.remaining, a.level
            FROM accounts a
            INNER JOIN students s ON s.id = a.student_id
            WHERE s.college_name = ?
        ");
        $stmt->execute([$college]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * جلب طالب واحد كامل مع حسابه
     */
    public static function singleStudent($student_id)
    {
        global $conn;

        // Auto create حساب إذا ما موجود
        $acc = AccountController::getAccount($student_id);

        $stmt = $conn->prepare("
            SELECT 
                s.*,
                a.total_fee, a.paid, a.remaining, a.level, a.due_date
            FROM students s
            LEFT JOIN accounts a ON a.student_id = s.id
            WHERE s.id = ?
        ");
        $stmt->execute([$student_id]);

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * جلب دفعات طالب واحد
     */
    public static function studentPayments($student_id)
    {
        global $conn;

        $stmt = $conn->prepare("
            SELECT *
            FROM payments 
            WHERE student_id = ?
            ORDER BY paid_at DESC
        ");
        $stmt->execute([$student_id]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * بيانات الرسم البياني
     */
    public static function chartSummary()
    {
        global $conn;

        return $conn->query("
            SELECT 
                SUM(total_fee) AS total,
                SUM(paid) AS paid,
                SUM(remaining) AS remaining
            FROM accounts
        ")->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * قائمة الكليات
     */
    public static function listColleges()
    {
        global $conn;

        return $conn->query("
            SELECT DISTINCT college_name 
            FROM students 
            WHERE college_name IS NOT NULL
        ")->fetchAll(PDO::FETCH_COLUMN);
    }

    /**
     * قائمة المراحل الدراسية
     */
    public static function listLevels()
    {
        return [
            "المرحلة الأولى",
            "المرحلة الثانية",
            "المرحلة الثالثة",
            "المرحلة الرابعة"
        ];
    }

    /**
     * قائمة الجنسين
     */
    public static function listGenders()
    {
        return ["ذكر", "أنثى"];
    }

    /**
     * HTML جاهز لـ PDF
     */
    public static function renderHTML($title, $rows)
    {
        ob_start();
        ?>
        <h2 style="text-align:center; font-family:'dejavusans';">
            <?= htmlspecialchars($title) ?>
        </h2>

        <table border="1" cellspacing="0" cellpadding="6" width="100%" 
               style="font-size:12px; font-family:'dejavusans'; text-align:center;">

            <thead style="background:#f0f0f0;">
                <tr>
                    <th>الاسم</th>
                    <th>الكلية</th>
                    <th>المرحلة</th>
                    <th>الجنس</th>
                    <th>الرسوم الكلية</th>
                    <th>المدفوع</th>
                    <th>المتبقي</th>
                </tr>
            </thead>

            <tbody>
                <?php foreach ($rows as $r): ?>
                <tr>
                    <td><?= $r['full_name'] ?></td>
                    <td><?= $r['college_name'] ?></td>
                    <td><?= $r['level'] ?></td>
                    <td><?= $r['gender'] ?></td>
                    <td><?= number_format($r['total_fee']) ?></td>
                    <td><?= number_format($r['paid']) ?></td>
                    <td><?= number_format($r['remaining']) ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>

        </table>
        <?php
        return ob_get_clean();
    }
    public static function getReport($filter)
{
    switch ($filter) {

        case "delayed":
            return self::lateStudents();

        case "male":
            return self::byGender("ذكر");

        case "female":
            return self::byGender("أنثى");

        case "level1":
            return self::byLevel("المرحلة الأولى");

        case "level2":
            return self::byLevel("المرحلة الثانية");

        case "level3":
            return self::byLevel("المرحلة الثالثة");

        case "level4":
            return self::byLevel("المرحلة الرابعة");

        default:
            return self::allPayments(); // عرض جميع الطلاب
    }
}
}
