<?php
// core/StudentController.php – B2Pro Ultra Edition

require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/Logger.php';
require_once __DIR__ . '/AccountController.php';
require_once __DIR__ . '/TimelineController.php';

class StudentController {

    /**
     * تنظيف النصوص قبل الحفظ (أمان)
     */
    private static function clean($v) {
        return trim(htmlspecialchars($v ?? '', ENT_QUOTES, 'UTF-8'));
    }

    /**
     * إنشاء طالب جديد
     */
    public static function create($data, $admin_id = null) {
        global $conn;

        // تنظيف البيانات
        foreach ($data as $k => $v) {
            $data[$k] = self::clean($v);
        }

        $sql = "INSERT INTO students (
            full_name, national_or_passport, phone, email, mother_name, birth_place, address, nearest_point, gender,
            guardian_phone1, guardian_phone2, student_photo, school_name, branch, exam_round, attempts, fail_years,
            exam_number, secret_number, subjects_count, total_score, average, admission_channel,
            university_name, college_name, study_type, app_username, app_password, dorm_request, status
        ) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";

        $stmt = $conn->prepare($sql);

        $ok = $stmt->execute([
            $data['full_name'],
            $data['national_or_passport'],
            $data['phone'],
            $data['email'],
            $data['mother_name'],
            $data['birth_place'],
            $data['address'],
            $data['nearest_point'],
            $data['gender'],
            $data['guardian_phone1'],
            $data['guardian_phone2'],
            $data['student_photo'],
            $data['school_name'],
            $data['branch'],
            $data['exam_round'],
            $data['attempts'],
            $data['fail_years'],
            $data['exam_number'],
            $data['secret_number'],
            $data['subjects_count'],
            $data['total_score'],
            $data['average'],
            $data['admission_channel'],
            $data['university_name'],
            $data['college_name'],
            $data['study_type'],
            $data['app_username'],
            $data['app_password'],
            $data['dorm_request'],
            'pending'
        ]);

        if ($ok) {

            $student_id = $conn->lastInsertId();

            // إنشاء حساب مالي فارغ تلقائيًا
            AccountController::getAccount($student_id);

            // سجل
            Logger::log($admin_id, $student_id, "CREATE_STUDENT", "تم إنشاء طالب جديد");
            TimelineController::addEvent($admin_id, $student_id, "CREATE_STUDENT", "إنشاء طالب جديد");

            return $student_id;
        }

        return false;
    }

    /**
     * جلب طالب حسب ID
     */
    public static function getById($id) {
        global $conn;

        $stmt = $conn->prepare("SELECT * FROM students WHERE id=? LIMIT 1");
        $stmt->execute([$id]);

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * جلب جميع الطلاب (مع حدود)
     */
    public static function getAll($limit = 200, $offset = 0) {
        global $conn;

        $stmt = $conn->prepare("SELECT * FROM students ORDER BY id DESC LIMIT ? OFFSET ?");
        $stmt->bindValue(1, (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(2, (int)$offset, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * تعديل بيانات طالب
     */
    public static function update($id, $data, $admin_id = null) {
        global $conn;

        // تنظيف
        foreach ($data as $k => $v) {
            $data[$k] = self::clean($v);
        }

        // معالجة تغيير الصورة (إن وجدت)
        if (!empty($data['student_photo_new'])) {
            $data['student_photo'] = $data['student_photo_new'];
        }

        $sql = "UPDATE students SET 
            full_name=?, national_or_passport=?, phone=?, email=?, mother_name=?, birth_place=?, address=?, nearest_point=?, gender=?,
            guardian_phone1=?, guardian_phone2=?, school_name=?, branch=?, exam_round=?, attempts=?, fail_years=?,
            exam_number=?, secret_number=?, subjects_count=?, total_score=?, average=?, admission_channel=?,
            university_name=?, college_name=?, study_type=?, app_username=?, app_password=?, dorm_request=?, status=?
            WHERE id=?";

        $stmt = $conn->prepare($sql);

        $ok = $stmt->execute([
            $data['full_name'],
            $data['national_or_passport'],
            $data['phone'],
            $data['email'],
            $data['mother_name'],
            $data['birth_place'],
            $data['address'],
            $data['nearest_point'],
            $data['gender'],
            $data['guardian_phone1'],
            $data['guardian_phone2'],
            $data['school_name'],
            $data['branch'],
            $data['exam_round'],
            $data['attempts'],
            $data['fail_years'],
            $data['exam_number'],
            $data['secret_number'],
            $data['subjects_count'],
            $data['total_score'],
            $data['average'],
            $data['admission_channel'],
            $data['university_name'],
            $data['college_name'],
            $data['study_type'],
            $data['app_username'],
            $data['app_password'],
            $data['dorm_request'],
            $data['status'],
            $id
        ]);

        if ($ok) {
            Logger::log($admin_id, $id, "UPDATE_STUDENT", "تم تعديل بيانات الطالب");
            TimelineController::addEvent($admin_id, $id, "UPDATE_STUDENT", "تعديل بيانات الطالب");
        }

        return $ok;
    }

    /**
     * حذف طالب مع البيانات المرتبطة به بالكامل
     */
    public static function delete($id, $admin_id = null) {
        global $conn;

        // حذف الحساب المالي
        $conn->prepare("DELETE FROM accounts WHERE student_id=?")->execute([$id]);

        // حذف الدفعات
        $conn->prepare("DELETE FROM payments WHERE student_id=?")->execute([$id]);

        // حذف الإشعارات
        $conn->prepare("DELETE FROM notifications WHERE student_id=?")->execute([$id]);

        // حذف السجلات
        $conn->prepare("DELETE FROM logs WHERE student_id=?")->execute([$id]);

        // أخيراً → حذف الطالب نفسه
        $stmt = $conn->prepare("DELETE FROM students WHERE id=?");
        $ok = $stmt->execute([$id]);

        if ($ok) {
            Logger::log($admin_id, $id, "DELETE_STUDENT", "تم حذف الطالب نهائياً");
            TimelineController::addEvent($admin_id, $id, "DELETE_STUDENT", "حذف كل بيانات الطالب");
        }

        return $ok;
    }

    /**
     * توثيق / تدقيق طالب
     */
    public static function verify($id, $admin_id = null) {
        global $conn;

        $stmt = $conn->prepare("UPDATE students SET status='verified' WHERE id=?");
        $ok = $stmt->execute([$id]);

        if ($ok) {
            Logger::log($admin_id, $id, "VERIFY_STUDENT", "تم تدقيق بيانات الطالب");
            TimelineController::addEvent($admin_id, $id, "VERIFY_STUDENT", "توثيق بيانات الطالب");
        }

        return $ok;
    }
}