<?php
session_start();
if (!isset($_SESSION["finance_user"])) {
    header("Location: login.php");
    exit;
}

require_once "../config/db.php";
require_once "../vendor/autoload.php";

/* ============================================================
   🔐 دالة حماية النصوص
============================================================ */
function safe($v){
    return htmlspecialchars(trim((string)$v), ENT_QUOTES, "UTF-8");
}

/* ============================================================
   📌 الموظف المستخرج للتقرير
============================================================ */
$employee = safe($_SESSION["finance_user"]);

/* ============================================================
   📌 استقبال الفلاتر
============================================================ */
$name_filter    = $_GET["name"] ?? "";
$college_filter = $_GET["college"] ?? "";
$study_filter   = $_GET["study"] ?? "";

/* ============================================================
   📌 بناء شروط الفلترة
============================================================ */
$where = "
    a.remaining > 0
    AND a.due_date IS NOT NULL
    AND a.due_date < CURDATE()
";

$params = [];

// FILTER — NAME
if (!empty($name_filter)) {
    $where .= " AND s.full_name LIKE ?";
    $params[] = "%$name_filter%";
}

// FILTER — COLLEGE
if (!empty($college_filter)) {
    $where .= " AND s.college_name = ?";
    $params[] = $college_filter;
}

// FILTER — STUDY TYPE
if (!empty($study_filter)) {
    $where .= " AND s.study_type = ?";
    $params[] = $study_filter;
}

/* ============================================================
   📌 جلب البيانات
============================================================ */
$sql = "
    SELECT 
        s.full_name,
        s.phone,
        s.college_name,
        s.study_type,
        a.remaining,
        a.due_date,
        DATEDIFF(CURDATE(), a.due_date) AS days_late
    FROM accounts a
    JOIN students s ON s.id = a.student_id
    WHERE $where
    ORDER BY days_late DESC
";

$stmt = $conn->prepare($sql);
$stmt->execute($params);
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* ============================================================
   📌 التاريخ والوقت بالعربي
============================================================ */
$days_ar = [
    "Monday" => "الاثنين",
    "Tuesday" => "الثلاثاء",
    "Wednesday" => "الأربعاء",
    "Thursday" => "الخميس",
    "Friday" => "الجمعة",
    "Saturday" => "السبت",
    "Sunday" => "الأحد"
];

$day_ar = $days_ar[date("l")];
$date = date("Y-m-d");
$time = date("h:i A");

/* ============================================================
   📌 تهيئة mPDF
============================================================ */
$tempDir = __DIR__ . '/../tmp';
if (!file_exists($tempDir)) mkdir($tempDir, 0777, true);

$mpdf = new \Mpdf\Mpdf([
    "mode" => "utf-8",
    "format" => "A4",
    "orientation" => "P",
    "tempDir" => $tempDir,
    "margin_left" => 10,
    "margin_right" => 10,
    "margin_top" => 45,
    "margin_bottom" => 15,
    "default_font" => "dejavusans",
]);

/* ============================================================
   📌 HEADER — الشعار بالوسط
============================================================ */
$logo = "../assets/img/logo.png";

$header = '
<div style="text-align:center;">
    <img src="'.$logo.'" style="width:85px; margin-bottom:4px;">
</div>

<div style="text-align:center; font-size:20px; font-weight:bold;">
جامعة النبراس الأهلية
</div>

<div style="text-align:center; font-size:14px; margin-top:3px;">
قسم الشؤون المالية — تقرير الطلاب المتأخرين عن السداد
</div>

<div style="text-align:center; font-size:12px; margin-top:2px;">
التاريخ: '.$date.' — الوقت: '.$time.' — اليوم: '.$day_ar.'
</div>
';

$mpdf->SetHTMLHeader($header);

/* ============================================================
   📌 FOOTER
============================================================ */
$mpdf->SetHTMLFooter('
<div style="border-top:1px solid #aaa; font-size:11px; padding-top:3px;" dir="rtl">
    <table width="100%">
        <tr>
            <td align="right">قسم الشؤون المالية – جامعة النبراس</td>
            <td align="left">صفحة {PAGENO} من {nb}</td>
        </tr>
    </table>
</div>
');

/* ============================================================
   📌 TABLE HTML
============================================================ */
$html = '
<style>
th {
    background:#374151;
    color:white;
    padding:8px;
    font-size:13px;
}
td {
    padding:8px;
    border:1px solid #ddd;
    font-size:13px;
}
tr:nth-child(even){ background:#f3f4f6; }
tr:nth-child(odd){ background:#ffffff; }
</style>

<table width="100%" dir="rtl" cellspacing="0" cellpadding="0">
<thead>
<tr>
    <th width="5%">#</th>
    <th width="20%">اسم الطالب</th>
    <th width="18%">القسم</th>
    <th width="10%">الدراسة</th>
    <th width="13%">الهاتف</th>
    <th width="15%">المبلغ المتأخر</th>
    <th width="10%">الأيام</th>
    <th width="12%">تاريخ الاستحقاق</th>
</tr>
</thead>
<tbody>
';

if (!$rows) {
    $html .= '<tr><td colspan="8" style="text-align:center;">لا يوجد طلاب ضمن الفلاتر المحددة</td></tr>';
} else {
    foreach ($rows as $i => $r) {
        $html .= "
        <tr>
            <td>".($i+1)."</td>
            <td>".safe($r['full_name'])."</td>
            <td>".safe($r['college_name'])."</td>
            <td>".safe($r['study_type'])."</td>
            <td>".safe($r['phone'])."</td>
            <td>".number_format($r['remaining'])."</td>
            <td>".safe($r['days_late'])." يوم</td>
            <td>".safe($r['due_date'])."</td>
        </tr>
        ";
    }
}

$html .= "</tbody></table>";

/* ============================================================
   📌 إخراج PDF
============================================================ */
$mpdf->WriteHTML($html);
$mpdf->Output("late_students_report.pdf", "I");
